# 17 "lexer.mll"
 
open Syntax
open Parser

(* Auxiliaries for the lexical analyzer *)

let brace_depth = ref 0
and comment_depth = ref 0

let in_pattern () = !brace_depth = 0 && !comment_depth = 0

exception Lexical_error of string * string * int * int

let string_buff = Buffer.create 256

let reset_string_buffer () = Buffer.clear string_buff

let store_string_char c = Buffer.add_char string_buff c

let get_stored_string () = Buffer.contents string_buff

let char_for_backslash = function
    'n' -> '\n'
  | 't' -> '\t'
  | 'b' -> '\b'
  | 'r' -> '\r'
  | c   -> c

let raise_lexical_error lexbuf msg =
  let p = Lexing.lexeme_start_p lexbuf in
  raise (Lexical_error (msg,
                        p.Lexing.pos_fname,
                        p.Lexing.pos_lnum,
                        p.Lexing.pos_cnum - p.Lexing.pos_bol + 1))
;;

let handle_lexical_error fn lexbuf =
  let p = Lexing.lexeme_start_p lexbuf in
  let line = p.Lexing.pos_lnum
  and column = p.Lexing.pos_cnum - p.Lexing.pos_bol + 1
  and file = p.Lexing.pos_fname
  in
  try
    fn lexbuf
  with Lexical_error (msg, "", 0, 0) ->
    raise(Lexical_error(msg, file, line, column))

let get_input_name () = Sys.argv.(Array.length Sys.argv - 1)

let warning lexbuf msg =
  let p = Lexing.lexeme_start_p lexbuf in
  Printf.eprintf "ocamllex warning:\nFile \"%s\", line %d, character %d: %s.\n"
    p.Lexing.pos_fname p.Lexing.pos_lnum
    (p.Lexing.pos_cnum - p.Lexing.pos_bol + 1) msg;
  flush stderr

let decimal_code  c d u =
  100 * (Char.code c - 48) + 10 * (Char.code d - 48) + (Char.code u - 48)

let char_for_hexadecimal_code d u =
  let d1 = Char.code d in
  let val1 = if d1 >= 97 then d1 - 87
             else if d1 >= 65 then d1 - 55
             else d1 - 48
  in
  let d2 = Char.code u in
  let val2 = if d2 >= 97 then d2 - 87
             else if d2 >= 65 then d2 - 55
             else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

let incr_loc lexbuf delta =
  let pos = lexbuf.Lexing.lex_curr_p in
  lexbuf.Lexing.lex_curr_p <- { pos with
    Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
    Lexing.pos_bol = pos.Lexing.pos_cnum - delta;
  }
;;

let update_loc lexbuf opt_file line =
  let pos = lexbuf.Lexing.lex_curr_p in
  let new_file = match opt_file with
                 | None -> pos.Lexing.pos_fname
                 | Some f -> f
  in
  lexbuf.Lexing.lex_curr_p <- { pos with
    Lexing.pos_fname = new_file;
    Lexing.pos_lnum = line;
    Lexing.pos_bol = pos.Lexing.pos_cnum;
  }
;;


# 97 "lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\229\255\230\255\232\255\233\255\234\255\236\255\237\255\
    \238\255\239\255\240\255\241\255\242\255\243\255\004\000\249\255\
    \218\000\170\001\002\000\116\000\254\255\005\000\126\000\079\001\
    \253\255\006\000\020\000\117\000\118\000\253\255\252\255\136\002\
    \007\000\248\255\244\255\245\001\041\001\008\000\247\255\051\001\
    \009\000\246\255\170\002\010\000\245\255\123\000\247\255\248\255\
    \249\255\241\002\255\255\250\255\019\003\089\001\253\255\018\000\
    \006\000\012\002\252\255\058\003\251\255\043\002\249\255\250\255\
    \251\255\252\255\253\255\010\000\010\000\255\255\254\255\161\002\
    \248\255\249\255\250\255\011\000\252\255\253\255\254\255\255\255\
    \251\255\090\002\016\000\017\000\090\003\018\000\114\003\155\002\
    \019\000\255\255\209\002\254\255\020\000\137\003\021\000";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\026\000\255\255\
    \005\000\004\000\020\000\024\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\011\000\011\000\011\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\008\000\255\255\255\255\005\000\005\000\255\255\001\000\
    \001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\006\000\006\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\007\000\255\255\255\255\255\255\255\255\
    \255\255\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\025\000\255\255\
    \000\000\025\000\026\000\025\000\028\000\000\000\000\000\034\000\
    \255\255\000\000\000\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\255\255\000\000\046\000\000\000\000\000\
    \000\000\051\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\255\255\000\000\255\255\000\000\062\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\000\000\000\000\072\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\082\000\255\255\255\255\085\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\255\255\255\255\255\255";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\021\000\020\000\000\000\021\000\021\000\021\000\056\000\
    \024\000\021\000\021\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\056\000\056\000\000\000\024\000\000\000\
    \021\000\255\255\015\000\019\000\000\000\021\000\056\000\014\000\
    \018\000\005\000\008\000\006\000\030\000\003\000\033\000\038\000\
    \041\000\044\000\056\000\070\000\069\000\080\000\028\000\091\000\
    \089\000\091\000\089\000\091\000\091\000\012\000\000\000\007\000\
    \000\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\010\000\000\000\009\000\004\000\017\000\
    \031\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\013\000\011\000\023\000\027\000\024\000\
    \029\000\000\000\255\255\255\255\000\000\047\000\000\000\027\000\
    \024\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\023\000\027\000\000\000\026\000\
    \000\000\000\000\000\000\000\000\000\000\050\000\027\000\000\000\
    \026\000\000\000\000\000\000\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\049\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\000\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \002\000\016\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \000\000\000\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\000\000\000\000\000\000\
    \000\000\016\000\000\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\000\000\000\000\000\000\
    \023\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\000\000\000\000\023\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\000\000\
    \000\000\000\000\000\000\048\000\000\000\000\000\255\255\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\000\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\000\000\000\000\000\000\
    \000\000\016\000\000\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\063\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\066\000\000\000\000\000\
    \000\000\000\000\065\000\068\000\000\000\067\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\083\000\000\000\000\000\000\000\
    \000\000\000\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\255\255\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\000\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\037\000\073\000\000\000\000\000\000\000\037\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\084\000\000\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\091\000\077\000\000\000\000\000\000\000\000\000\
    \076\000\075\000\000\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\037\000\000\000\000\000\000\000\
    \000\000\000\000\037\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\000\000\000\000\000\000\000\000\000\000\037\000\000\000\
    \000\000\000\000\037\000\056\000\037\000\000\000\055\000\000\000\
    \035\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\000\000\000\000\054\000\000\000\000\000\000\000\000\000\
    \054\000\000\000\000\000\000\000\079\000\000\000\078\000\000\000\
    \000\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\064\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\054\000\000\000\000\000\
    \000\000\000\000\000\000\054\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\255\255\000\000\000\000\000\000\000\000\054\000\
    \000\000\000\000\000\000\054\000\088\000\054\000\000\000\000\000\
    \000\000\052\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\000\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\087\000\087\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\000\000\000\000\000\000\000\000\
    \000\000\091\000\000\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\074\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \000\000\000\000\086\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\000\000\000\000\021\000\056\000\
    \025\000\021\000\021\000\025\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\055\000\055\000\255\255\026\000\255\255\
    \000\000\026\000\000\000\000\000\255\255\021\000\056\000\000\000\
    \000\000\000\000\000\000\000\000\018\000\000\000\032\000\037\000\
    \040\000\043\000\055\000\067\000\068\000\075\000\026\000\082\000\
    \083\000\085\000\088\000\092\000\094\000\000\000\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \014\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\019\000\027\000\027\000\
    \028\000\255\255\027\000\028\000\255\255\045\000\255\255\022\000\
    \022\000\255\255\255\255\022\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\019\000\027\000\255\255\027\000\
    \255\255\255\255\255\255\255\255\255\255\045\000\022\000\255\255\
    \022\000\255\255\255\255\255\255\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\045\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\016\000\255\255\255\255\014\000\255\255\025\000\255\255\
    \255\255\255\255\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\026\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\255\255\255\255\255\255\
    \255\255\016\000\255\255\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\255\255\255\255\255\255\
    \023\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\255\255\255\255\023\000\
    \255\255\255\255\255\255\255\255\255\255\027\000\028\000\255\255\
    \255\255\255\255\255\255\045\000\255\255\255\255\022\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\023\000\023\000\
    \023\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\255\255\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\017\000\016\000\016\000\016\000\016\000\016\000\016\000\
    \016\000\016\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\255\255\255\255\255\255\
    \255\255\017\000\255\255\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\061\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\061\000\255\255\255\255\
    \255\255\255\255\061\000\061\000\255\255\061\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\081\000\255\255\255\255\255\255\
    \255\255\255\255\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\081\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\255\255\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\031\000\071\000\255\255\255\255\255\255\031\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\081\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\087\000\071\000\255\255\255\255\255\255\255\255\
    \071\000\071\000\255\255\087\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\087\000\087\000\087\000\255\255\255\255\255\255\
    \255\255\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\031\000\255\255\255\255\255\255\
    \255\255\255\255\031\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\255\255\255\255\255\255\255\255\255\255\031\000\255\255\
    \255\255\255\255\031\000\049\000\031\000\255\255\049\000\255\255\
    \031\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\255\255\255\255\049\000\255\255\255\255\255\255\255\255\
    \049\000\255\255\255\255\255\255\071\000\255\255\071\000\255\255\
    \255\255\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\061\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\049\000\255\255\255\255\
    \255\255\255\255\255\255\049\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\081\000\255\255\255\255\255\255\255\255\049\000\
    \255\255\255\255\255\255\049\000\084\000\049\000\255\255\255\255\
    \255\255\049\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\255\255\059\000\059\000\059\000\059\000\059\000\
    \059\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\255\255\255\255\255\255\255\255\
    \255\255\086\000\255\255\059\000\059\000\059\000\059\000\059\000\
    \059\000\071\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\086\000\086\000\086\000\086\000\086\000\
    \086\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\093\000\093\000\093\000\093\000\093\000\093\000\
    \255\255\255\255\084\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\093\000\093\000\093\000\093\000\093\000\093\000\
    \255\255\049\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\084\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\036\000\
    \012\000\000\000\000\000\002\000\000\000\027\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\000\000\000\000\000\000\000\000\002\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\039\000\
    \039\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\019\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\019\000\026\000\049\000\055\000\026\000\049\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \019\000\255\255\026\000\000\000\027\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\022\000\023\000\255\255\255\255\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\023\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\023\000\023\000\023\000\023\000\
    \023\000\023\000\023\000\023\000\023\000\023\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \026\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 8 (-1) ;   __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 121 "lexer.mll"
    ( main lexbuf )
# 560 "lexer.ml"

  | 1 ->
# 123 "lexer.mll"
    ( incr_loc lexbuf 0;
      main lexbuf )
# 566 "lexer.ml"

  | 2 ->
let
# 125 "lexer.mll"
                                   num
# 572 "lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 126 "lexer.mll"
                                      name
# 577 "lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 128 "lexer.mll"
    ( update_loc lexbuf name (int_of_string num);
      main lexbuf
    )
# 583 "lexer.ml"

  | 3 ->
# 132 "lexer.mll"
    ( comment_depth := 1;
      handle_lexical_error comment lexbuf;
      main lexbuf )
# 590 "lexer.ml"

  | 4 ->
# 135 "lexer.mll"
        ( Tunderscore )
# 595 "lexer.ml"

  | 5 ->
# 137 "lexer.mll"
    ( match Lexing.lexeme lexbuf with
        "rule" -> Trule
      | "parse" -> Tparse
      | "shortest" -> Tparse_shortest
      | "and" -> Tand
      | "eof" -> Teof
      | "let" -> Tlet
      | "as"  -> Tas
      | s -> Tident s )
# 608 "lexer.ml"

  | 6 ->
# 147 "lexer.mll"
    ( reset_string_buffer();
      handle_lexical_error string lexbuf;
      Tstring(get_stored_string()) )
# 615 "lexer.ml"

  | 7 ->
# 152 "lexer.mll"
    ( Tchar(Char.code(Lexing.lexeme_char lexbuf 1)) )
# 620 "lexer.ml"

  | 8 ->
# 154 "lexer.mll"
    ( Tchar(Char.code(char_for_backslash (Lexing.lexeme_char lexbuf 2))) )
# 625 "lexer.ml"

  | 9 ->
let
# 155 "lexer.mll"
                           c
# 631 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2)
and
# 155 "lexer.mll"
                                            d
# 636 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3)
and
# 155 "lexer.mll"
                                                             u
# 641 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 4) in
# 156 "lexer.mll"
    ( let v = decimal_code c d u in
      if v > 255 then
        raise_lexical_error lexbuf
          (Printf.sprintf "illegal escape sequence \\%c%c%c" c d u)
      else
        Tchar v )
# 650 "lexer.ml"

  | 10 ->
let
# 163 "lexer.mll"
                                     d
# 656 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3)
and
# 163 "lexer.mll"
                                                                      u
# 661 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 4) in
# 164 "lexer.mll"
       ( Tchar(Char.code(char_for_hexadecimal_code d u)) )
# 665 "lexer.ml"

  | 11 ->
let
# 165 "lexer.mll"
                   c
# 671 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 166 "lexer.mll"
    ( raise_lexical_error lexbuf
        (Printf.sprintf "illegal escape sequence \\%c" c)
    )
# 677 "lexer.ml"

  | 12 ->
# 170 "lexer.mll"
    ( let p = Lexing.lexeme_end_p lexbuf in
      let n1 = p.Lexing.pos_cnum
      and l1 = p.Lexing.pos_lnum
      and s1 = p.Lexing.pos_bol in
      brace_depth := 1;
      let n2 = handle_lexical_error action lexbuf in
      Taction({start_pos = n1; end_pos = n2;
               start_line = l1; start_col = n1 - s1}) )
# 689 "lexer.ml"

  | 13 ->
# 178 "lexer.mll"
         ( Tequal )
# 694 "lexer.ml"

  | 14 ->
# 179 "lexer.mll"
         ( Tor )
# 699 "lexer.ml"

  | 15 ->
# 180 "lexer.mll"
         ( Tlbracket )
# 704 "lexer.ml"

  | 16 ->
# 181 "lexer.mll"
         ( Trbracket )
# 709 "lexer.ml"

  | 17 ->
# 182 "lexer.mll"
         ( Tstar )
# 714 "lexer.ml"

  | 18 ->
# 183 "lexer.mll"
         ( Tmaybe )
# 719 "lexer.ml"

  | 19 ->
# 184 "lexer.mll"
         ( Tplus )
# 724 "lexer.ml"

  | 20 ->
# 185 "lexer.mll"
         ( Tlparen )
# 729 "lexer.ml"

  | 21 ->
# 186 "lexer.mll"
         ( Trparen )
# 734 "lexer.ml"

  | 22 ->
# 187 "lexer.mll"
         ( Tcaret )
# 739 "lexer.ml"

  | 23 ->
# 188 "lexer.mll"
         ( Tdash )
# 744 "lexer.ml"

  | 24 ->
# 189 "lexer.mll"
         ( Tsharp )
# 749 "lexer.ml"

  | 25 ->
# 190 "lexer.mll"
         ( Tend )
# 754 "lexer.ml"

  | 26 ->
# 192 "lexer.mll"
    ( raise_lexical_error lexbuf
        ("illegal character " ^ String.escaped(Lexing.lexeme lexbuf))
    )
# 761 "lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 2 (-1) ;   __ocaml_lex_string_rec lexbuf 45
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 200 "lexer.mll"
    ( () )
# 772 "lexer.ml"

  | 1 ->
let
# 201 "lexer.mll"
                                                            spaces
# 778 "lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 202 "lexer.mll"
    ( incr_loc lexbuf (String.length spaces);
      string lexbuf )
# 783 "lexer.ml"

  | 2 ->
let
# 204 "lexer.mll"
                               c
# 789 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 205 "lexer.mll"
    ( store_string_char(char_for_backslash c);
      string lexbuf )
# 794 "lexer.ml"

  | 3 ->
let
# 207 "lexer.mll"
                       c
# 800 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 207 "lexer.mll"
                                        d
# 805 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2)
and
# 207 "lexer.mll"
                                                          u
# 810 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3) in
# 208 "lexer.mll"
    ( let v = decimal_code c d u in
      if in_pattern () && v > 255 then
       warning lexbuf
        (Printf.sprintf
          "illegal backslash escape in string: `\\%c%c%c'" c d u) ;
      store_string_char (Char.chr v);
      string lexbuf )
# 820 "lexer.ml"

  | 4 ->
let
# 215 "lexer.mll"
                                          d
# 826 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2)
and
# 215 "lexer.mll"
                                                                           u
# 831 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3) in
# 216 "lexer.mll"
    ( store_string_char (char_for_hexadecimal_code d u) ;
      string lexbuf )
# 836 "lexer.ml"

  | 5 ->
let
# 218 "lexer.mll"
               c
# 842 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 219 "lexer.mll"
    (if in_pattern () then
       warning lexbuf
        (Printf.sprintf "illegal backslash escape in string: `\\%c'" c) ;
      store_string_char '\\' ;
      store_string_char c ;
      string lexbuf )
# 851 "lexer.ml"

  | 6 ->
# 226 "lexer.mll"
    ( raise(Lexical_error("unterminated string", "", 0, 0)) )
# 856 "lexer.ml"

  | 7 ->
# 228 "lexer.mll"
    ( store_string_char '\010';
      incr_loc lexbuf 0;
      string lexbuf )
# 863 "lexer.ml"

  | 8 ->
let
# 231 "lexer.mll"
         c
# 869 "lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 232 "lexer.mll"
    ( store_string_char c;
      string lexbuf )
# 874 "lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 61
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 243 "lexer.mll"
    ( incr comment_depth; comment lexbuf )
# 885 "lexer.ml"

  | 1 ->
# 245 "lexer.mll"
    ( decr comment_depth;
      if !comment_depth = 0 then () else comment lexbuf )
# 891 "lexer.ml"

  | 2 ->
# 248 "lexer.mll"
    ( reset_string_buffer();
      string lexbuf;
      reset_string_buffer();
      comment lexbuf )
# 899 "lexer.ml"

  | 3 ->
# 253 "lexer.mll"
    ( skip_char lexbuf ;
      comment lexbuf )
# 905 "lexer.ml"

  | 4 ->
# 256 "lexer.mll"
    ( raise(Lexical_error("unterminated comment", "", 0, 0)) )
# 910 "lexer.ml"

  | 5 ->
# 258 "lexer.mll"
    ( incr_loc lexbuf 0;
      comment lexbuf )
# 916 "lexer.ml"

  | 6 ->
# 261 "lexer.mll"
    ( comment lexbuf )
# 921 "lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and action lexbuf =
    __ocaml_lex_action_rec lexbuf 71
and __ocaml_lex_action_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 265 "lexer.mll"
    ( incr brace_depth;
      action lexbuf )
# 933 "lexer.ml"

  | 1 ->
# 268 "lexer.mll"
    ( decr brace_depth;
      if !brace_depth = 0 then Lexing.lexeme_start lexbuf else action lexbuf )
# 939 "lexer.ml"

  | 2 ->
# 271 "lexer.mll"
    ( reset_string_buffer();
      handle_lexical_error string lexbuf;
      reset_string_buffer();
      action lexbuf )
# 947 "lexer.ml"

  | 3 ->
# 276 "lexer.mll"
    ( skip_char lexbuf ;
      action lexbuf )
# 953 "lexer.ml"

  | 4 ->
# 279 "lexer.mll"
    ( comment_depth := 1;
      comment lexbuf;
      action lexbuf )
# 960 "lexer.ml"

  | 5 ->
# 283 "lexer.mll"
    ( raise (Lexical_error("unterminated action", "", 0, 0)) )
# 965 "lexer.ml"

  | 6 ->
# 285 "lexer.mll"
    ( incr_loc lexbuf 0;
      action lexbuf )
# 971 "lexer.ml"

  | 7 ->
# 288 "lexer.mll"
    ( action lexbuf )
# 976 "lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_action_rec lexbuf __ocaml_lex_state

and skip_char lexbuf =
    __ocaml_lex_skip_char_rec lexbuf 81
and __ocaml_lex_skip_char_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 292 "lexer.mll"
     ( incr_loc lexbuf 1;
     )
# 988 "lexer.ml"

  | 1 ->
# 299 "lexer.mll"
     (())
# 993 "lexer.ml"

  | 2 ->
# 301 "lexer.mll"
       (())
# 998 "lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_skip_char_rec lexbuf __ocaml_lex_state

;;

